<?php
class ApnsPHP_Log_Embedded implements ApnsPHP_Log_Interface
{
	/**
	 * Logs a message.
	 *
	 * @param  $sMessage @type string The message.
	 */
	public function log($sMessage)
	{
		printf("%s ApnsPHP[%d]: %s\n",
			date('r'), getmypid(), trim($sMessage)
		);
	}
}

class ApnsPHP_Log_Error implements ApnsPHP_Log_Interface
{
	/**
	 * Logs a message.
	 *
	 * @param  $sMessage @type string The message.
	 */
	public function log($sMessage)
	{
		error_log(" ApnsPHP: ".trim($sMessage));
	}
}

interface ApnsPHP_Log_Interface
{
	/**
	 * Logs a message.
	 *
	 * @param  $sMessage @type string The message.
	 */
	public function log($sMessage);
}


class ApnsPHP_Log_Silent implements ApnsPHP_Log_Interface
{
	/**
	 * Logs a message.
	 *
	 * @param  $sMessage @type string The message.
	 */
	public function log($sMessage)
	{

	}
}

class ApnsPHP_Push_Exception extends ApnsPHP_Exception
{
}


class ApnsPHP_Push_Server_Exception extends ApnsPHP_Push_Exception
{
}


/**
 * @file
 * ApnsPHP_Push_Server class definition.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * @defgroup ApnsPHP_Push_Server Server
 * @ingroup ApnsPHP_Push
 */

/**
 * The Push Notification Server Provider.
 *
 * The class manages multiple Push Notification Providers and an inter-process message
 * queue. This class is useful to parallelize and speed-up send activities to Apple
 * Push Notification service.
 *
 * @ingroup ApnsPHP_Push_Server
 */
 class ApnsPHP_Push_Server extends ApnsPHP_Push
 {
	 const MAIN_LOOP_USLEEP = 200000; /**< @type integer Main loop sleep time in micro seconds. */
	 const SHM_SIZE = 524288; /**< @type integer Shared memory size in bytes useful to store message queues. */
	 const SHM_MESSAGES_QUEUE_KEY_START = 1000; /**< @type integer Message queue start identifier for messages. For every process 1 is added to this number. */
	 const SHM_ERROR_MESSAGES_QUEUE_KEY = 999; /**< @type integer Message queue identifier for not delivered messages. */
 
	 protected $_nProcesses = 3; /**< @type integer The number of processes to start. */
	 protected $_aPids = array(); /**< @type array Array of process PIDs. */
	 protected $_nParentPid; /**< @type integer The parent process id. */
	 protected $_nCurrentProcess; /**< @type integer Cardinal process number (0, 1, 2, ...). */
	 protected $_nRunningProcesses; /**< @type integer The number of running processes. */
 
	 protected $_hShm; /**< @type resource Shared memory. */
	 protected $_hSem; /**< @type resource Semaphore. */
 
	 /**
	  * Constructor.
	  *
	  * @param  $nEnvironment @type integer Environment.
	  * @param  $sProviderCertificateFile @type string Provider certificate file
	  *         with key (Bundled PEM).
	  * @throws ApnsPHP_Push_Server_Exception if is unable to
	  *         get Shared Memory Segment or Semaphore ID.
	  */
	 public function __construct($nEnvironment, $sProviderCertificateFile)
	 {
		 parent::__construct($nEnvironment, $sProviderCertificateFile);
 
		 $this->_nParentPid = posix_getpid();
		 $this->_hShm = shm_attach(mt_rand(), self::SHM_SIZE);
		 if ($this->_hShm === false) {
			 throw new ApnsPHP_Push_Server_Exception(
				 'Unable to get shared memory segment'
			 );
		 }
 
		 $this->_hSem = sem_get(mt_rand());
		 if ($this->_hSem === false) {
			 throw new ApnsPHP_Push_Server_Exception(
				 'Unable to get semaphore id'
			 );
		 }
 
		 register_shutdown_function(array($this, 'onShutdown'));
 
		 pcntl_signal(SIGCHLD, array($this, 'onChildExited'));
		 foreach(array(SIGTERM, SIGQUIT, SIGINT) as $nSignal) {
			 pcntl_signal($nSignal, array($this, 'onSignal'));
		 }
	 }
 
	 /**
	  * Checks if the server is running and calls signal handlers for pending signals.
	  *
	  * Example:
	  * @code
	  * while ($Server->run()) {
	  *     // do somethings...
	  *     usleep(200000);
	  * }
	  * @endcode
	  *
	  * @return @type boolean True if the server is running.
	  */
	 public function run()
	 {
		 pcntl_signal_dispatch();
		 return $this->_nRunningProcesses > 0;
	 }
 
	 /**
	  * Waits until a forked process has exited and decreases the current running
	  * process number.
	  */
	 public function onChildExited()
	 {
		 while (pcntl_waitpid(-1, $nStatus, WNOHANG) > 0) {
			 $this->_nRunningProcesses--;
		 }
	 }
 
	 /**
	  * When a child (not the parent) receive a signal of type TERM, QUIT or INT
	  * exits from the current process and decreases the current running process number.
	  *
	  * @param  $nSignal @type integer Signal number.
	  */
	 public function onSignal($nSignal)
	 {
		 switch ($nSignal) {
			 case SIGTERM:
			 case SIGQUIT:
			 case SIGINT:
				 if (($nPid = posix_getpid()) != $this->_nParentPid) {
					 $this->_log("INFO: Child $nPid received signal #{$nSignal}, shutdown...");
					 $this->_nRunningProcesses--;
					 exit(0);
				 }
				 break;
			 default:
				 $this->_log("INFO: Ignored signal #{$nSignal}.");
				 break;
		 }
	 }
 
	 /**
	  * When the parent process exits, cleans shared memory and semaphore.
	  *
	  * This is called using 'register_shutdown_function' pattern.
	  * @see http://php.net/register_shutdown_function
	  */
	 public function onShutdown()
	 {
		 if (posix_getpid() == $this->_nParentPid) {
			 $this->_log('INFO: Parent shutdown, cleaning memory...');
			 @shm_remove($this->_hShm) && @shm_detach($this->_hShm);
			 @sem_remove($this->_hSem);
		 }
	 }
 
	 /**
	  * Set the total processes to start, default is 3.
	  *
	  * @param  $nProcesses @type integer Processes to start up.
	  */
	 public function setProcesses($nProcesses)
	 {
		 $nProcesses = (int)$nProcesses;
		 if ($nProcesses <= 0) {
			 return;
		 }
		 $this->_nProcesses = $nProcesses;
	 }
 
	 /**
	  * Starts the server forking all processes and return immediately.
	  *
	  * Every forked process is connected to Apple Push Notification Service on start
	  * and enter on the main loop.
	  */
	 public function start()
	 {
		 for ($i = 0; $i < $this->_nProcesses; $i++) {
			 $this->_nCurrentProcess = $i;
			 $this->_aPids[$i] = $nPid = pcntl_fork();
			 if ($nPid == -1) {
				 $this->_log('WARNING: Could not fork');
			 } else if ($nPid > 0) {
				 // Parent process
				 $this->_log("INFO: Forked process PID {$nPid}");
				 $this->_nRunningProcesses++;
			 } else {
				 // Child process
				 try {
					 parent::connect();
				 } catch (ApnsPHP_Exception $e) {
					 $this->_log('ERROR: ' . $e->getMessage() . ', exiting...');
					 exit(1);
				 }
				 $this->_mainLoop();
				 parent::disconnect();
				 exit(0);
			 }
		 }
	 }
 
	 /**
	  * Adds a message to the inter-process message queue.
	  *
	  * Messages are added to the queues in a round-robin fashion starting from the
	  * first process to the last.
	  *
	  * @param  $message @type ApnsPHP_Message The message.
	  */
	 public function add(ApnsPHP_Message $message)
	 {
		 static $n = 0;
		 if ($n >= $this->_nProcesses) {
			 $n = 0;
		 }
		 sem_acquire($this->_hSem);
		 $aQueue = $this->_getQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $n);
		 $aQueue[] = $message;
		 $this->_setQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $n, $aQueue);
		 sem_release($this->_hSem);
		 $n++;
	 }
 
	 /**
	  * Returns messages in the message queue.
	  *
	  * When a message is successful sent or reached the maximum retry time is removed
	  * from the message queue and inserted in the Errors container. Use the getErrors()
	  * method to retrive messages with delivery error(s).
	  *
	  * @param  $bEmpty @type boolean @optional Empty message queue.
	  * @return @type array Array of messages left on the queue.
	  */
	 public function getQueue($bEmpty = true)
	 {
		 $aRet = array();
		 sem_acquire($this->_hSem);
		 for ($i = 0; $i < $this->_nProcesses; $i++) {
			 $aRet = array_merge($aRet, $this->_getQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $i));
			 if ($bEmpty) {
				 $this->_setQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $i);
			 }
		 }
		 sem_release($this->_hSem);
		 return $aRet;
	 }
 
	 /**
	  * Returns messages not delivered to the end user because one (or more) error
	  * occurred.
	  *
	  * @param  $bEmpty @type boolean @optional Empty message container.
	  * @return @type array Array of messages not delivered because one or more errors
	  *         occurred.
	  */
	 public function getErrors($bEmpty = true)
	 {
		 sem_acquire($this->_hSem);
		 $aRet = $this->_getQueue(self::SHM_ERROR_MESSAGES_QUEUE_KEY);
		 if ($bEmpty) {
			 $this->_setQueue(self::SHM_ERROR_MESSAGES_QUEUE_KEY, 0, array());
		 }
		 sem_release($this->_hSem);
		 return $aRet;
	 }
 
	 /**
	  * The process main loop.
	  *
	  * During the main loop: the per-process error queue is read and the common error message
	  * container is populated; the per-process message queue is spooled (message from
	  * this queue is added to ApnsPHP_Push queue and delivered).
	  */
	 protected function _mainLoop()
	 {
		 while (true) {
			 pcntl_signal_dispatch();
 
			 if (posix_getppid() != $this->_nParentPid) {
				 $this->_log("INFO: Parent process {$this->_nParentPid} died unexpectedly, exiting...");
				 break;
			 }
 
			 sem_acquire($this->_hSem);
			 $this->_setQueue(self::SHM_ERROR_MESSAGES_QUEUE_KEY, 0,
				 array_merge($this->_getQueue(self::SHM_ERROR_MESSAGES_QUEUE_KEY), parent::getErrors())
			 );
 
			 $aQueue = $this->_getQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $this->_nCurrentProcess);
			 foreach($aQueue as $message) {
				 parent::add($message);
			 }
			 $this->_setQueue(self::SHM_MESSAGES_QUEUE_KEY_START, $this->_nCurrentProcess);
			 sem_release($this->_hSem);
 
			 $nMessages = count($aQueue);
			 if ($nMessages > 0) {
				 $this->_log('INFO: Process ' . ($this->_nCurrentProcess + 1) . " has {$nMessages} messages, sending...");
				 parent::send();
			 } else {
				 usleep(self::MAIN_LOOP_USLEEP);
			 }
		 }
	 }
 
	 /**
	  * Returns the queue from the shared memory.
	  *
	  * @param  $nQueueKey @type integer The key of the queue stored in the shared
	  *         memory.
	  * @param  $nProcess @type integer @optional The process cardinal number.
	  * @return @type array Array of messages from the queue.
	  */
	 protected function _getQueue($nQueueKey, $nProcess = 0)
	 {
		 if (!shm_has_var($this->_hShm, $nQueueKey + $nProcess)) {
			 return array();
		 }
		 return shm_get_var($this->_hShm, $nQueueKey + $nProcess);
	 }
 
	 /**
	  * Store the queue into the shared memory.
	  *
	  * @param  $nQueueKey @type integer The key of the queue to store in the shared
	  *         memory.
	  * @param  $nProcess @type integer @optional The process cardinal number.
	  * @param  $aQueue @type array @optional The queue to store into shared memory.
	  *         The default value is an empty array, useful to empty the queue.
	  * @return @type boolean True on success, false otherwise.
	  */
	 protected function _setQueue($nQueueKey, $nProcess = 0, $aQueue = array())
	 {
		 if (!is_array($aQueue)) {
			 $aQueue = array();
		 }
		 return shm_put_var($this->_hShm, $nQueueKey + $nProcess, $aQueue);
	 }
 }

/**
 * @file
 * ApnsPHP_Abstract class definition.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * @mainpage
 *
 * @li ApnsPHP on GitHub: https://github.com/immobiliare/ApnsPHP
 */

/**
 * @defgroup ApplePushNotificationService ApnsPHP
 */

/**
 * Abstract class: this is the superclass for all Apple Push Notification Service
 * classes.
 *
 * This class is responsible for the connection to the Apple Push Notification Service
 * and Feedback.
 *
 * @ingroup ApplePushNotificationService
 * @see http://tinyurl.com/ApplePushNotificationService
 */
abstract class ApnsPHP_Abstract
{
	const ENVIRONMENT_PRODUCTION = 0; /**< @type integer Production environment. */
	const ENVIRONMENT_SANDBOX = 1; /**< @type integer Sandbox environment. */

	const DEVICE_BINARY_SIZE = 32; /**< @type integer Device token length. */

	const WRITE_INTERVAL = 10000; /**< @type integer Default write interval in micro seconds. */
	const CONNECT_RETRY_INTERVAL = 1000000; /**< @type integer Default connect retry interval in micro seconds. */
	const SOCKET_SELECT_TIMEOUT = 1000000; /**< @type integer Default socket select timeout in micro seconds. */

	protected $_aServiceURLs = array(); /**< @type array Container for service URLs environments. */

	protected $_nEnvironment; /**< @type integer Active environment. */

	protected $_nConnectTimeout; /**< @type integer Connect timeout in seconds. */
	protected $_nConnectRetryTimes = 3; /**< @type integer Connect retry times. */

	protected $_sProviderCertificateFile; /**< @type string Provider certificate file with key (Bundled PEM). */
	protected $_sProviderCertificatePassphrase; /**< @type string Provider certificate passphrase. */
	protected $_sRootCertificationAuthorityFile; /**< @type string Root certification authority file. */

	protected $_nWriteInterval; /**< @type integer Write interval in micro seconds. */
	protected $_nConnectRetryInterval; /**< @type integer Connect retry interval in micro seconds. */
	protected $_nSocketSelectTimeout; /**< @type integer Socket select timeout in micro seconds. */

	protected $_logger; /**< @type ApnsPHP_Log_Interface Logger. */

	protected $_hSocket; /**< @type resource SSL Socket. */

	/**
	 * Constructor.
	 *
	 * @param  $nEnvironment @type integer Environment.
	 * @param  $sProviderCertificateFile @type string Provider certificate file
	 *         with key (Bundled PEM).
	 * @throws ApnsPHP_Exception if the environment is not
	 *         sandbox or production or the provider certificate file is not readable.
	 */
	public function __construct($nEnvironment, $sProviderCertificateFile)
	{
		if ($nEnvironment != self::ENVIRONMENT_PRODUCTION && $nEnvironment != self::ENVIRONMENT_SANDBOX) {
			throw new ApnsPHP_Exception(
				"Invalid environment '{$nEnvironment}'"
			);
		}
		$this->_nEnvironment = $nEnvironment;

		if (!is_readable($sProviderCertificateFile)) {
			throw new ApnsPHP_Exception(
				"Unable to read certificate file '{$sProviderCertificateFile}'"
			);
		}
		$this->_sProviderCertificateFile = $sProviderCertificateFile;

		$this->_nConnectTimeout = ini_get("default_socket_timeout");
		$this->_nWriteInterval = self::WRITE_INTERVAL;
		$this->_nConnectRetryInterval = self::CONNECT_RETRY_INTERVAL;
		$this->_nSocketSelectTimeout = self::SOCKET_SELECT_TIMEOUT;
	}

	/**
	 * Set the Logger instance to use for logging purpose.
	 *
	 * The default logger is ApnsPHP_Log_Embedded, an instance
	 * of ApnsPHP_Log_Interface that simply print to standard
	 * output log messages.
	 *
	 * To set a custom logger you have to implement ApnsPHP_Log_Interface
	 * and use setLogger, otherwise standard logger will be used.
	 *
	 * @see ApnsPHP_Log_Interface
	 * @see ApnsPHP_Log_Embedded
	 *
	 * @param  $logger @type ApnsPHP_Log_Interface Logger instance.
	 * @throws ApnsPHP_Exception if Logger is not an instance
	 *         of ApnsPHP_Log_Interface.
	 */
	public function setLogger(ApnsPHP_Log_Interface $logger)
	{
		if (!is_object($logger)) {
			throw new ApnsPHP_Exception(
				"The logger should be an instance of 'ApnsPHP_Log_Interface'"
			);
		}
		if (!($logger instanceof ApnsPHP_Log_Interface)) {
			throw new ApnsPHP_Exception(
				"Unable to use an instance of '" . get_class($logger) . "' as logger: " .
				"a logger must implements ApnsPHP_Log_Interface."
			);
		}
		$this->_logger = $logger;
	}

	/**
	 * Get the Logger instance.
	 *
	 * @return @type ApnsPHP_Log_Interface Current Logger instance.
	 */
	public function getLogger()
	{
		return $this->_logger;
	}

	/**
	 * Set the Provider Certificate passphrase.
	 *
	 * @param  $sProviderCertificatePassphrase @type string Provider Certificate
	 *         passphrase.
	 */
	public function setProviderCertificatePassphrase($sProviderCertificatePassphrase)
	{
		$this->_sProviderCertificatePassphrase = $sProviderCertificatePassphrase;
	}

	/**
	 * Set the Root Certification Authority file.
	 *
	 * Setting the Root Certification Authority file automatically set peer verification
	 * on connect.
	 *
	 * @see http://tinyurl.com/GeneralProviderRequirements
	 * @see http://www.entrust.net/
	 * @see https://www.entrust.net/downloads/root_index.cfm
	 *
	 * @param  $sRootCertificationAuthorityFile @type string Root Certification
	 *         Authority file.
	 * @throws ApnsPHP_Exception if Root Certification Authority
	 *         file is not readable.
	 */
	public function setRootCertificationAuthority($sRootCertificationAuthorityFile)
	{
		if (!is_readable($sRootCertificationAuthorityFile)) {
			throw new ApnsPHP_Exception(
				"Unable to read Certificate Authority file '{$sRootCertificationAuthorityFile}'"
			);
		}
		$this->_sRootCertificationAuthorityFile = $sRootCertificationAuthorityFile;
	}

	/**
	 * Get the Root Certification Authority file path.
	 *
	 * @return @type string Current Root Certification Authority file path.
	 */
	public function getCertificateAuthority()
	{
		return $this->_sRootCertificationAuthorityFile;
	}

	/**
	 * Set the write interval.
	 *
	 * After each socket write operation we are sleeping for this 
	 * time interval. To speed up the sending operations, use Zero
	 * as parameter but some messages may be lost.
	 *
	 * @param  $nWriteInterval @type integer Write interval in micro seconds.
	 */
	public function setWriteInterval($nWriteInterval)
	{
		$this->_nWriteInterval = (int)$nWriteInterval;
	}

	/**
	 * Get the write interval.
	 *
	 * @return @type integer Write interval in micro seconds.
	 */
	public function getWriteInterval()
	{
		return $this->_nWriteInterval;
	}

	/**
	 * Set the connection timeout.
	 *
	 * The default connection timeout is the PHP internal value "default_socket_timeout".
	 * @see http://php.net/manual/en/filesystem.configuration.php
	 *
	 * @param  $nTimeout @type integer Connection timeout in seconds.
	 */
	public function setConnectTimeout($nTimeout)
	{
		$this->_nConnectTimeout = (int)$nTimeout;
	}

	/**
	 * Get the connection timeout.
	 *
	 * @return @type integer Connection timeout in seconds.
	 */
	public function getConnectTimeout()
	{
		return $this->_nConnectTimeout;
	}

	/**
	 * Set the connect retry times value.
	 *
	 * If the client is unable to connect to the server retries at least for this
	 * value. The default connect retry times is 3.
	 *
	 * @param  $nRetryTimes @type integer Connect retry times.
	 */
	public function setConnectRetryTimes($nRetryTimes)
	{
		$this->_nConnectRetryTimes = (int)$nRetryTimes;
	}

	/**
	 * Get the connect retry time value.
	 *
	 * @return @type integer Connect retry times.
	 */
	public function getConnectRetryTimes()
	{
		return $this->_nConnectRetryTimes;
	}

	/**
	 * Set the connect retry interval.
	 *
	 * If the client is unable to connect to the server retries at least for ConnectRetryTimes
	 * and waits for this value between each attempts.
	 *
	 * @see setConnectRetryTimes
	 *
	 * @param  $nRetryInterval @type integer Connect retry interval in micro seconds.
	 */
	public function setConnectRetryInterval($nRetryInterval)
	{
		$this->_nConnectRetryInterval = (int)$nRetryInterval;
	}

	/**
	 * Get the connect retry interval.
	 *
	 * @return @type integer Connect retry interval in micro seconds.
	 */
	public function getConnectRetryInterval()
	{
		return $this->_nConnectRetryInterval;
	}

	/**
	 * Set the TCP socket select timeout.
	 *
	 * After writing to socket waits for at least this value for read stream to
	 * change status.
	 *
	 * In Apple Push Notification protocol there isn't a real-time
	 * feedback about the correctness of notifications pushed to the server; so after
	 * each write to server waits at least SocketSelectTimeout. If, during this
	 * time, the read stream change its status and socket received an end-of-file
	 * from the server the notification pushed to server was broken, the server
	 * has closed the connection and the client needs to reconnect.
	 *
	 * @see http://php.net/stream_select
	 *
	 * @param  $nSelectTimeout @type integer Socket select timeout in micro seconds.
	 */
	public function setSocketSelectTimeout($nSelectTimeout)
	{
		$this->_nSocketSelectTimeout = (int)$nSelectTimeout;
	}

	/**
	 * Get the TCP socket select timeout.
	 *
	 * @return @type integer Socket select timeout in micro seconds.
	 */
	public function getSocketSelectTimeout()
	{
		return $this->_nSocketSelectTimeout;
	}

	/**
	 * Connects to Apple Push Notification service server.
	 *
	 * Retries ConnectRetryTimes if unable to connect and waits setConnectRetryInterval
	 * between each attempts.
	 *
	 * @see setConnectRetryTimes
	 * @see setConnectRetryInterval
	 * @throws ApnsPHP_Exception if is unable to connect after
	 *         ConnectRetryTimes.
	 */
	public function connect()
	{
		$bConnected = false;
		$nRetry = 0;
		while (!$bConnected) {
			try {
				$bConnected = $this->_connect();
			} catch (ApnsPHP_Exception $e) {
				$this->_log('ERROR: ' . $e->getMessage());
				if ($nRetry >= $this->_nConnectRetryTimes) {
					throw $e;
				} else {
					$this->_log(
						"INFO: Retry to connect (" . ($nRetry+1) .
						"/{$this->_nConnectRetryTimes})..."
					);
					usleep($this->_nConnectRetryInterval);
				}
			}
			$nRetry++;
		}
	}

	/**
	 * Disconnects from Apple Push Notifications service server.
	 *
	 * @return @type boolean True if successful disconnected.
	 */
	public function disconnect()
	{
		if (is_resource($this->_hSocket)) {
			$this->_log('INFO: Disconnected.');
			return fclose($this->_hSocket);
		}
		return false;
	}

	/**
	 * Connects to Apple Push Notification service server.
	 *
	 * @throws ApnsPHP_Exception if is unable to connect.
	 * @return @type boolean True if successful connected.
	 */
	protected function _connect()
	{
		$sURL = $this->_aServiceURLs[$this->_nEnvironment];
		unset($aURLs);

		$this->_log("INFO: Trying {$sURL}...");

		/**
		 * @see http://php.net/manual/en/context.ssl.php
		 */
		$streamContext = stream_context_create(array('ssl' => array(
			'verify_peer' => isset($this->_sRootCertificationAuthorityFile),
			'cafile' => $this->_sRootCertificationAuthorityFile,
			'local_cert' => $this->_sProviderCertificateFile
		)));

		if (!empty($this->_sProviderCertificatePassphrase)) {
			stream_context_set_option($streamContext, 'ssl',
				'passphrase', $this->_sProviderCertificatePassphrase);
		}

		$this->_hSocket = @stream_socket_client($sURL, $nError, $sError,
			$this->_nConnectTimeout, STREAM_CLIENT_CONNECT, $streamContext);

		if (!$this->_hSocket) {
			throw new ApnsPHP_Exception(
				"Unable to connect to '{$sURL}': {$sError} ({$nError})"
			);
		}

		stream_set_blocking($this->_hSocket, 0);
		stream_set_write_buffer($this->_hSocket, 0);

		$this->_log("INFO: Connected to {$sURL}.");

		return true;
	}

	/**
	 * Logs a message through the Logger.
	 *
	 * @param  $sMessage @type string The message.
	 */
	protected function _log($sMessage)
	{
		if (!isset($this->_logger)) {
			//$this->_logger = new ApnsPHP_Log_Embedded();
			$this->_logger = new ApnsPHP_Log_Silent();
		}
		$this->_logger->log($sMessage);
	}
}


/**
 * @file
 * Autoload stuff.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * This function is automatically called in case you are trying to use a
 * class/interface which hasn't been defined yet. By calling this function the
 * scripting engine is given a last chance to load the class before PHP
 * fails with an error.
 *
 * @see http://php.net/__autoload
 * @see http://php.net/spl_autoload_register
 *
 * @param  $sClassName @type string The class name.
 * @throws Exception if class name is empty, the current path is empty or class
 *         file does not exists or file was loaded but class name was not found.
 */
 function ApnsPHP_Autoload($sClassName)
 {
     if (empty($sClassName)) {
         throw new Exception('Class name is empty');
     }
 
     $sPath = dirname(dirname(__FILE__));
     if (empty($sPath)) {
         throw new Exception('Current path is empty');
     }
 
     $sFile = sprintf('%s%s%s.php',
         $sPath, DIRECTORY_SEPARATOR,
         str_replace('_', DIRECTORY_SEPARATOR, $sClassName)
     );
     if (is_file($sFile) && is_readable($sFile)) {
         require_once $sFile;
     }
 }
 
 // If your code has an existing __autoload function then this function must be explicitly registered on the __autoload stack.
 // (PHP Documentation for spl_autoload_register [@see http://php.net/spl_autoload_register])
 if (function_exists('__autoload')) {
     spl_autoload_register('__autoload');
 }
 spl_autoload_register('ApnsPHP_Autoload');

 /**
 * Exception class.
 *
 * @ingroup ApplePushNotificationService
 */
class ApnsPHP_Exception extends Exception
{
}

/**
 * @file
 * ApnsPHP_Feedback class definition.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * @defgroup ApnsPHP_Feedback Feedback
 * @ingroup ApplePushNotificationService
 */

/**
 * The Feedback Service client.
 *
 * Apple Push Notification Service includes a feedback service that APNs continually
 * updates with a per-application list of devices for which there were failed-delivery
 * attempts. Providers should periodically query the feedback service to get the
 * list of device tokens for their applications, each of which is identified by
 * its topic. Then, after verifying that the application hasn’t recently been re-registered
 * on the identified devices, a provider should stop sending notifications to these
 * devices.
 *
 * @ingroup ApnsPHP_Feedback
 * @see http://tinyurl.com/ApplePushNotificationFeedback
 */
 class ApnsPHP_Feedback extends ApnsPHP_Abstract
 {
     const TIME_BINARY_SIZE = 4; /**< @type integer Timestamp binary size in bytes. */
     const TOKEN_LENGTH_BINARY_SIZE = 2; /**< @type integer Token length binary size in bytes. */
 
     protected $_aServiceURLs = array(
         'tls://feedback.push.apple.com:2196', // Production environment
         'tls://feedback.sandbox.push.apple.com:2196' // Sandbox environment
     ); /**< @type array Feedback URLs environments. */
 
     protected $_aFeedback; /**< @type array Feedback container. */
 
     /**
      * Receives feedback tuples from Apple Push Notification Service feedback.
      *
      * Every tuple (array) contains:
      * @li @c timestamp indicating when the APNs determined that the application
      *     no longer exists on the device. This value represents the seconds since
      *     1970, anchored to UTC. You should use the timestamp to determine if the
      *     application on the device re-registered with your service since the moment
      *     the device token was recorded on the feedback service. If it hasn’t,
      *     you should cease sending push notifications to the device.
      * @li @c tokenLength The length of the device token (usually 32 bytes).
      * @li @c deviceToken The device token.
      *
      * @return @type array Array of feedback tuples (array).
      */
     public function receive()
     {
         $nFeedbackTupleLen = self::TIME_BINARY_SIZE + self::TOKEN_LENGTH_BINARY_SIZE + self::DEVICE_BINARY_SIZE;
 
         $this->_aFeedback = array();
         $sBuffer = '';
         while (!feof($this->_hSocket)) {
             $this->_log('INFO: Reading...');
             $sBuffer .= $sCurrBuffer = fread($this->_hSocket, 8192);
             $nCurrBufferLen = strlen($sCurrBuffer);
             if ($nCurrBufferLen > 0) {
                 $this->_log("INFO: {$nCurrBufferLen} bytes read.");
             }
             unset($sCurrBuffer, $nCurrBufferLen);
 
             $nBufferLen = strlen($sBuffer);
             if ($nBufferLen >= $nFeedbackTupleLen) {
                 $nFeedbackTuples = floor($nBufferLen / $nFeedbackTupleLen);
                 for ($i = 0; $i < $nFeedbackTuples; $i++) {
                     $sFeedbackTuple = substr($sBuffer, 0, $nFeedbackTupleLen);
                     $sBuffer = substr($sBuffer, $nFeedbackTupleLen);
                     $this->_aFeedback[] = $aFeedback = $this->_parseBinaryTuple($sFeedbackTuple);
                     $this->_log(sprintf("INFO: New feedback tuple: timestamp=%d (%s), tokenLength=%d, deviceToken=%s.",
                         $aFeedback['timestamp'], date('Y-m-d H:i:s', $aFeedback['timestamp']),
                         $aFeedback['tokenLength'], $aFeedback['deviceToken']
                     ));
                     unset($aFeedback);
                 }
             }
 
             $read = array($this->_hSocket);
             $null = NULL;
             $nChangedStreams = stream_select($read, $null, $null, 0, $this->_nSocketSelectTimeout);
             if ($nChangedStreams === false) {
                 $this->_log('WARNING: Unable to wait for a stream availability.');
                 break;
             }
         }
         return $this->_aFeedback;
     }
 
     /**
      * Parses binary tuples.
      *
      * @param  $sBinaryTuple @type string A binary tuple to parse.
      * @return @type array Array with timestamp, tokenLength and deviceToken keys.
      */
     protected function _parseBinaryTuple($sBinaryTuple)
     {
         return unpack('Ntimestamp/ntokenLength/H*deviceToken', $sBinaryTuple);
     }
 }
 
 /**
 * @file
 * ApnsPHP_Message class definition.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * @defgroup ApnsPHP_Message Message
 * @ingroup ApplePushNotificationService
 */

/**
 * The Push Notification Message.
 *
 * The class represents a message to be delivered to an end user device.
 * Notification Service.
 *
 * @ingroup ApnsPHP_Message
 * @see http://tinyurl.com/ApplePushNotificationPayload
 */
class ApnsPHP_Message
{
	const PAYLOAD_MAXIMUM_SIZE = 2048; /**< @type integer The maximum size allowed for a notification payload. */
	const APPLE_RESERVED_NAMESPACE = 'aps'; /**< @type string The Apple-reserved aps namespace. */

	protected $_bAutoAdjustLongPayload = true; /**< @type boolean If the JSON payload is longer than maximum allowed size, shorts message text. */

	protected $_aDeviceTokens = array(); /**< @type array Recipients device tokens. */

	protected $_sText; /**< @type string Alert message to display to the user. */
	protected $_sTitle; /**< @type string Alert title to display to the user. */
	protected $_nBadge; /**< @type integer Number to badge the application icon with. */
	protected $_sSound; /**< @type string Sound to play. */
	protected $_sCategory; /**< @type string notification category. */
	protected $_bContentAvailable; /**< @type boolean True to initiates the Newsstand background download. @see http://tinyurl.com/ApplePushNotificationNewsstand */
	protected $_bMutableContent; /**< @type boolean True to activate mutable content key support for ios10 rich notifications. @see https://developer.apple.com/reference/usernotifications/unnotificationserviceextension */

	protected $_aCustomProperties; /**< @type mixed Custom properties container. */

	protected $_nExpiryValue = 604800; /**< @type integer That message will expire in 604800 seconds (86400 * 7, 7 days) if not successful delivered. */

	protected $_mCustomIdentifier; /**< @type mixed Custom message identifier. */

	/**
	 * Constructor.
	 *
	 * @param  $sDeviceToken @type string @optional Recipients device token.
	 */
	public function __construct($sDeviceToken = null)
	{
		if (isset($sDeviceToken)) {
			$this->addRecipient($sDeviceToken);
		}
	}

	/**
	 * Add a recipient device token.
	 *
	 * @param  $sDeviceToken @type string Recipients device token.
	 * @throws ApnsPHP_Message_Exception if the device token
	 *         is not well formed.
	 */
	public function addRecipient($sDeviceToken)
	{
		if (!preg_match('~^[a-f0-9]{64}$~i', $sDeviceToken)) {
			throw new ApnsPHP_Message_Exception(
				"Invalid device token '{$sDeviceToken}'"
			);
		}
		$this->_aDeviceTokens[] = $sDeviceToken;
	}

	/**
	 * Get a recipient.
	 *
	 * @param  $nRecipient @type integer @optional Recipient number to return.
	 * @throws ApnsPHP_Message_Exception if no recipient number
	 *         exists.
	 * @return @type string The recipient token at index $nRecipient.
	 */
	public function getRecipient($nRecipient = 0)
	{
		if (!isset($this->_aDeviceTokens[$nRecipient])) {
			throw new ApnsPHP_Message_Exception(
				"No recipient at index '{$nRecipient}'"
			);
		}
		return $this->_aDeviceTokens[$nRecipient];
	}

	/**
	 * Get the number of recipients.
	 *
	 * @return @type integer Recipient's number.
	 */
	public function getRecipientsNumber()
	{
		return count($this->_aDeviceTokens);
	}

	/**
	 * Get all recipients.
	 *
	 * @return @type array Array of all recipients device token.
	 */
	public function getRecipients()
	{
		return $this->_aDeviceTokens;
	}

	/**
	 * Set the alert message to display to the user.
	 *
	 * @param  $sText @type string An alert message to display to the user.
	 */
	public function setText($sText)
	{
		$this->_sText = $sText;
	}

	/**
	 * Get the alert message to display to the user.
	 *
	 * @return @type string The alert message to display to the user.
	 */
	public function getText()
	{
		return $this->_sText;
	}

	/**
	 * Set the alert title to display to the user.  This will be BOLD text on the top of the push message. If
	 * this title is not set - only the _sText will be used in the alert without bold text. 
	 *
	 * @param  $sTitle @type string An alert title to display to the user.
	 */
	public function setTitle($sTitle)
	{
	    $this->_sTitle = $sTitle;
	}
	
	/**
	 * Get the alert title to display to the user.
	 *
	 * @return @type string The alert title to display to the user.
	 */
	public function getTitle()
	{
	    return $this->_sTitle;
	}
	
	/**
	 * Set the number to badge the application icon with.
	 *
	 * @param  $nBadge @type integer A number to badge the application icon with.
	 * @throws ApnsPHP_Message_Exception if badge is not an
	 *         integer.
	 */
	public function setBadge($nBadge)
	{
		if (!is_int($nBadge)) {
			throw new ApnsPHP_Message_Exception(
				"Invalid badge number '{$nBadge}'"
			);
		}
		$this->_nBadge = $nBadge;
	}

	/**
	 * Get the number to badge the application icon with.
	 *
	 * @return @type integer The number to badge the application icon with.
	 */
	public function getBadge()
	{
		return $this->_nBadge;
	}

	/**
	 * Set the sound to play.
	 *
	 * @param  $sSound @type string @optional A sound to play ('default sound' is
	 *         the default sound).
	 */
	public function setSound($sSound = 'default')
	{
		$this->_sSound = $sSound;
	}

	/**
	 * Get the sound to play.
	 *
	 * @return @type string The sound to play.
	 */
	public function getSound()
	{
		return $this->_sSound;
	}
	
	/**
	 * Set the category of notification
	 *
	 * @param  $sCategory @type string @optional A category for ios8 notification actions.
	 */
	public function setCategory($sCategory = '')
	{
		$this->_sCategory = $sCategory;
	}

	/**
	 * Get the category of notification
	 *
	 * @return @type string The notification category
	 */
	public function getCategory()
	{
		return $this->_sCategory;
	}

	/**
	 * Initiates the Newsstand background download.
	 * @see http://tinyurl.com/ApplePushNotificationNewsstand
	 *
	 * @param  $bContentAvailable @type boolean True to initiates the Newsstand background download.
	 * @throws ApnsPHP_Message_Exception if ContentAvailable is not a
	 *         boolean.
	 */
	public function setContentAvailable($bContentAvailable = true)
	{
		if (!is_bool($bContentAvailable)) {
			throw new ApnsPHP_Message_Exception(
				"Invalid content-available value '{$bContentAvailable}'"
			);
		}
		$this->_bContentAvailable = $bContentAvailable ? true : null;
	}

	/**
	 * Get if should initiates the Newsstand background download.
	 *
	 * @return @type boolean Initiates the Newsstand background download property.
	 */
	public function getContentAvailable()
	{
		return $this->_bContentAvailable;
	}

	/**
	 * Set the mutable-content key for Notification Service Extensions on iOS10
	 * @see https://developer.apple.com/reference/usernotifications/unnotificationserviceextension
	 *
	 * @param  $bMutableContent @type boolean True to enable flag
	 * @throws ApnsPHP_Message_Exception if MutableContent is not a
	 *         boolean.
	 */
	public function setMutableContent($bMutableContent = true)
	{
		if (!is_bool($bMutableContent)) {
			throw new ApnsPHP_Message_Exception(
				"Invalid mutable-content value '{$bMutableContent}'"
			);
		}
		$this->_bMutableContent = $bMutableContent ? true : null;
	}

	/**
	 * Get if should set the mutable-content ios10 rich notifications flag
	 *
	 * @return @type boolean mutable-content ios10 rich notifications flag
	 */
	public function getMutableContent()
	{
		return $this->_bMutableContent;
	}

	/**
	 * Set a custom property.
	 *
	 * @param  $sName @type string Custom property name.
	 * @param  $mValue @type mixed Custom property value.
	 * @throws ApnsPHP_Message_Exception if custom property name is not outside
	 *         the Apple-reserved 'aps' namespace.
	 */
	public function setCustomProperty($sName, $mValue)
	{
		if (trim($sName) == self::APPLE_RESERVED_NAMESPACE) {
			throw new ApnsPHP_Message_Exception(
				"Property name '" . self::APPLE_RESERVED_NAMESPACE . "' can not be used for custom property."
			);
		}
		$this->_aCustomProperties[trim($sName)] = $mValue;
	}

	/**
	 * Get the first custom property name.
	 *
	 * @deprecated Use getCustomPropertyNames() instead.
	 *
	 * @return @type string The first custom property name.
	 */
	public function getCustomPropertyName()
	{
		if (!is_array($this->_aCustomProperties)) {
			return;
		}
		$aKeys = array_keys($this->_aCustomProperties);
		return $aKeys[0];
	}

	/**
	 * Get the first custom property value.
	 *
	 * @deprecated Use getCustomProperty() instead.
	 *
	 * @return @type mixed The first custom property value.
	 */
	public function getCustomPropertyValue()
	{
		if (!is_array($this->_aCustomProperties)) {
			return;
		}
		$aKeys = array_keys($this->_aCustomProperties);
		return $this->_aCustomProperties[$aKeys[0]];
	}

	/**
	 * Get all custom properties names.
	 *
	 * @return @type array All properties names.
	 */
	public function getCustomPropertyNames()
	{
		if (!is_array($this->_aCustomProperties)) {
			return array();
		}
		return array_keys($this->_aCustomProperties);
	}

	/**
	 * Get the custom property value.
	 *
	 * @param  $sName @type string Custom property name.
	 * @throws ApnsPHP_Message_Exception if no property exists with the specified
	 *         name.
	 * @return @type string The custom property value.
	 */
	public function getCustomProperty($sName)
	{
		if (!array_key_exists($sName, $this->_aCustomProperties)) {
			throw new ApnsPHP_Message_Exception(
				"No property exists with the specified name '{$sName}'."
			);
		}
		return $this->_aCustomProperties[$sName];
	}

	/**
	 * Set the auto-adjust long payload value.
	 *
	 * @param  $bAutoAdjust @type boolean If true a long payload is shorted cutting
	 *         long text value.
	 */
	public function setAutoAdjustLongPayload($bAutoAdjust)
	{
		$this->_bAutoAdjustLongPayload = (boolean)$bAutoAdjust;
	}

	/**
	 * Get the auto-adjust long payload value.
	 *
	 * @return @type boolean The auto-adjust long payload value.
	 */
	public function getAutoAdjustLongPayload()
	{
		return $this->_bAutoAdjustLongPayload;
	}

	/**
	 * PHP Magic Method. When an object is "converted" to a string, JSON-encoded
	 * payload is returned.
	 *
	 * @return @type string JSON-encoded payload.
	 */
	public function __toString()
	{
		try {
			$sJSONPayload = $this->getPayload();
		} catch (ApnsPHP_Message_Exception $e) {
			$sJSONPayload = '';
		}
		return $sJSONPayload;
	}

	/**
	 * Get the payload dictionary.
	 * For more information on push titles see : https://stackoverflow.com/questions/40647061/bold-or-other-formatting-in-ios-push-notification
	 * @return @type array The payload dictionary.
	 */
	protected function _getPayload()
	{
		$aPayload[self::APPLE_RESERVED_NAMESPACE] = array();

		if (isset($this->_sText)) {
		    if (isset($this->_sTitle) && strlen($this->_sTitle) > 0) {
		        // if the title is set, use it 
		        $aPayload[self::APPLE_RESERVED_NAMESPACE]['alert'] = array();
		        $aPayload[self::APPLE_RESERVED_NAMESPACE]['alert']['title'] =  (string)$this->_sTitle;
		        $aPayload[self::APPLE_RESERVED_NAMESPACE]['alert']['body'] = (string)$this->_sText;
		    } else {
		        // if the title is not set, use the standard alert message format
		        $aPayload[self::APPLE_RESERVED_NAMESPACE]['alert'] = (string)$this->_sText;
		    }
		}
		
		if (isset($this->_nBadge) && $this->_nBadge >= 0) {
			$aPayload[self::APPLE_RESERVED_NAMESPACE]['badge'] = (int)$this->_nBadge;
		}
		if (isset($this->_sSound)) {
			$aPayload[self::APPLE_RESERVED_NAMESPACE]['sound'] = (string)$this->_sSound;
		}
		if (isset($this->_bContentAvailable)) {
			$aPayload[self::APPLE_RESERVED_NAMESPACE]['content-available'] = (int)$this->_bContentAvailable;
		}
		if (isset($this->_bMutableContent)) {
			$aPayload[self::APPLE_RESERVED_NAMESPACE]['mutable-content'] = (int)$this->_bMutableContent;
		}
		if (isset($this->_sCategory)) {
			$aPayload[self::APPLE_RESERVED_NAMESPACE]['category'] = (string)$this->_sCategory;
		}

		if (is_array($this->_aCustomProperties)) {
			foreach($this->_aCustomProperties as $sPropertyName => $mPropertyValue) {
				$aPayload[$sPropertyName] = $mPropertyValue;
			}
		}

		return $aPayload;
	}

	/**
	 * Convert the message in a JSON-encoded payload.
	 *
	 * @throws ApnsPHP_Message_Exception if payload is longer than maximum allowed
	 *         size and AutoAdjustLongPayload is disabled.
	 * @return @type string JSON-encoded payload.
	 */
	public function getPayload()
	{
		//$sJSON = json_encode($this->_getPayload(), defined('JSON_UNESCAPED_UNICODE') ? JSON_UNESCAPED_UNICODE : 0);
		$sJSON = json_encode($this->_getPayload());
		if (!defined('JSON_UNESCAPED_UNICODE') && function_exists('mb_convert_encoding')) {
			$sJSON = preg_replace_callback(
				'~\\\\u([0-9a-f]{4})~i',
				create_function('$aMatches', 'return mb_convert_encoding(pack("H*", $aMatches[1]), "UTF-8", "UTF-16");'),
				$sJSON);
		}

		$sJSONPayload = str_replace(
			'"' . self::APPLE_RESERVED_NAMESPACE . '":[]',
			'"' . self::APPLE_RESERVED_NAMESPACE . '":{}',
			$sJSON
		);
		$nJSONPayloadLen = strlen($sJSONPayload);

		if ($nJSONPayloadLen > self::PAYLOAD_MAXIMUM_SIZE) {
			if ($this->_bAutoAdjustLongPayload) {
				$nMaxTextLen = $nTextLen = strlen($this->_sText) - ($nJSONPayloadLen - self::PAYLOAD_MAXIMUM_SIZE);
				if ($nMaxTextLen > 0) {
					while (strlen($this->_sText = mb_substr($this->_sText, 0, --$nTextLen, 'UTF-8')) > $nMaxTextLen);
					return $this->getPayload();
				} else {
					throw new ApnsPHP_Message_Exception(
						"JSON Payload is too long: {$nJSONPayloadLen} bytes. Maximum size is " .
						self::PAYLOAD_MAXIMUM_SIZE . " bytes. The message text can not be auto-adjusted."
					);
				}
			} else {
				throw new ApnsPHP_Message_Exception(
					"JSON Payload is too long: {$nJSONPayloadLen} bytes. Maximum size is " .
					self::PAYLOAD_MAXIMUM_SIZE . " bytes"
				);
			}
		}

		return $sJSONPayload;
	}

	/**
	 * Set the expiry value.
	 *
	 * @param  $nExpiryValue @type integer This message will expire in N seconds
	 *         if not successful delivered.
	 */
	public function setExpiry($nExpiryValue)
	{
		if (!is_int($nExpiryValue)) {
			throw new ApnsPHP_Message_Exception(
				"Invalid seconds number '{$nExpiryValue}'"
			);
		}
		$this->_nExpiryValue = $nExpiryValue;
	}

	/**
	 * Get the expiry value.
	 *
	 * @return @type integer The expire message value (in seconds).
	 */
	public function getExpiry()
	{
		return $this->_nExpiryValue;
	}

	/**
	 * Set the custom message identifier.
	 *
	 * The custom message identifier is useful to associate a push notification
	 * to a DB record or an User entry for example. The custom message identifier
	 * can be retrieved in case of error using the getCustomIdentifier()
	 * method of an entry retrieved by the getErrors() method.
	 * This custom identifier, if present, is also used in all status message by
	 * the ApnsPHP_Push class.
	 *
	 * @param  $mCustomIdentifier @type mixed The custom message identifier.
	 */
	public function setCustomIdentifier($mCustomIdentifier)
	{
		$this->_mCustomIdentifier = $mCustomIdentifier;
	}

	/**
	 * Get the custom message identifier.
	 *
	 * @return @type mixed The custom message identifier.
	 */
	public function getCustomIdentifier()
	{
		return $this->_mCustomIdentifier;
	}
}

/**
 * @file
 * ApnsPHP_Push class definition.
 *
 * LICENSE
 *
 * This source file is subject to the new BSD license that is bundled
 * with this package in the file LICENSE.txt.
 * It is also available through the world-wide-web at this URL:
 * http://code.google.com/p/apns-php/wiki/License
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to aldo.armiento@gmail.com so we can send you a copy immediately.
 *
 * @author (C) 2010 Aldo Armiento (aldo.armiento@gmail.com)
 * @version $Id$
 */

/**
 * @defgroup ApnsPHP_Push Push
 * @ingroup ApplePushNotificationService
 */

/**
 * The Push Notification Provider.
 *
 * The class manages a message queue and sends notifications payload to Apple Push
 * Notification Service.
 *
 * @ingroup ApnsPHP_Push
 */
 class ApnsPHP_Push extends ApnsPHP_Abstract
 {
     const COMMAND_PUSH = 1; /**< @type integer Payload command. */
 
     const ERROR_RESPONSE_SIZE = 6; /**< @type integer Error-response packet size. */
     const ERROR_RESPONSE_COMMAND = 8; /**< @type integer Error-response command code. */
 
     const STATUS_CODE_INTERNAL_ERROR = 999; /**< @type integer Status code for internal error (not Apple). */
 
     protected $_aErrorResponseMessages = array(
         0   => 'No errors encountered',
         1   => 'Processing error',
         2   => 'Missing device token',
         3   => 'Missing topic',
         4   => 'Missing payload',
         5   => 'Invalid token size',
         6   => 'Invalid topic size',
         7   => 'Invalid payload size',
         8   => 'Invalid token',
         self::STATUS_CODE_INTERNAL_ERROR => 'Internal error'
     ); /**< @type array Error-response messages. */
 
     protected $_nSendRetryTimes = 3; /**< @type integer Send retry times. */
 
     protected $_aServiceURLs = array(
         'tls://gateway.push.apple.com:2195', // Production environment
         'tls://gateway.sandbox.push.apple.com:2195' // Sandbox environment
     ); /**< @type array Service URLs environments. */
 
     protected $_aMessageQueue = array(); /**< @type array Message queue. */
     protected $_aErrors = array(); /**< @type array Error container. */
 
     /**
      * Set the send retry times value.
      *
      * If the client is unable to send a payload to to the server retries at least
      * for this value. The default send retry times is 3.
      *
      * @param  $nRetryTimes @type integer Send retry times.
      */
     public function setSendRetryTimes($nRetryTimes)
     {
         $this->_nSendRetryTimes = (int)$nRetryTimes;
     }
 
     /**
      * Get the send retry time value.
      *
      * @return @type integer Send retry times.
      */
     public function getSendRetryTimes()
     {
         return $this->_nSendRetryTimes;
     }
 
     /**
      * Adds a message to the message queue.
      *
      * @param  $message @type ApnsPHP_Message The message.
      */
     public function add(ApnsPHP_Message $message)
     {
         $sMessagePayload = $message->getPayload();
         $nRecipients = $message->getRecipientsNumber();
 
         $nMessageQueueLen = count($this->_aMessageQueue);
         for ($i = 0; $i < $nRecipients; $i++) {
             $nMessageID = $nMessageQueueLen + $i + 1;
             $this->_aMessageQueue[$nMessageID] = array(
                 'MESSAGE' => $message,
                 'BINARY_NOTIFICATION' => $this->_getBinaryNotification(
                     $message->getRecipient($i),
                     $sMessagePayload,
                     $nMessageID,
                     $message->getExpiry()
                 ),
                 'ERRORS' => array()
             );
         }
     }
 
     /**
      * Sends all messages in the message queue to Apple Push Notification Service.
      *
      * @throws ApnsPHP_Push_Exception if not connected to the
      *         service or no notification queued.
      */
     public function send()
     {
         if (!$this->_hSocket) {
             throw new ApnsPHP_Push_Exception(
                 'Not connected to Push Notification Service'
             );
         }
 
         if (empty($this->_aMessageQueue)) {
             throw new ApnsPHP_Push_Exception(
                 'No notifications queued to be sent'
             );
         }
 
         $this->_aErrors = array();
         $nRun = 1;
         while (($nMessages = count($this->_aMessageQueue)) > 0) {
             $this->_log("INFO: Sending messages queue, run #{$nRun}: $nMessages message(s) left in queue.");
 
             $bError = false;
             foreach($this->_aMessageQueue as $k => &$aMessage) {
                 if (function_exists('pcntl_signal_dispatch')) {
                     pcntl_signal_dispatch();
                 }
 
                 $message = $aMessage['MESSAGE'];
                 $sCustomIdentifier = (string)$message->getCustomIdentifier();
                 $sCustomIdentifier = sprintf('[custom identifier: %s]', empty($sCustomIdentifier) ? 'unset' : $sCustomIdentifier);
 
                 $nErrors = 0;
                 if (!empty($aMessage['ERRORS'])) {
                     foreach($aMessage['ERRORS'] as $aError) {
                         if ($aError['statusCode'] == 0) {
                             $this->_log("INFO: Message ID {$k} {$sCustomIdentifier} has no error ({$aError['statusCode']}), removing from queue...");
                             $this->_removeMessageFromQueue($k);
                             continue 2;
                         } else if ($aError['statusCode'] > 1 && $aError['statusCode'] <= 8) {
                             $this->_log("WARNING: Message ID {$k} {$sCustomIdentifier} has an unrecoverable error ({$aError['statusCode']}), removing from queue without retrying...");
                             $this->_removeMessageFromQueue($k, true);
                             continue 2;
                         }
                     }
                     if (($nErrors = count($aMessage['ERRORS'])) >= $this->_nSendRetryTimes) {
                         $this->_log(
                             "WARNING: Message ID {$k} {$sCustomIdentifier} has {$nErrors} errors, removing from queue..."
                         );
                         $this->_removeMessageFromQueue($k, true);
                         continue;
                     }
                 }
 
                 $nLen = strlen($aMessage['BINARY_NOTIFICATION']);
                 $this->_log("STATUS: Sending message ID {$k} {$sCustomIdentifier} (" . ($nErrors + 1) . "/{$this->_nSendRetryTimes}): {$nLen} bytes.");
 
                 $aErrorMessage = null;
                 if ($nLen !== ($nWritten = (int)@fwrite($this->_hSocket, $aMessage['BINARY_NOTIFICATION']))) {
                     $aErrorMessage = array(
                         'identifier' => $k,
                         'statusCode' => self::STATUS_CODE_INTERNAL_ERROR,
                         'statusMessage' => sprintf('%s (%d bytes written instead of %d bytes)',
                             $this->_aErrorResponseMessages[self::STATUS_CODE_INTERNAL_ERROR], $nWritten, $nLen
                         )
                     );
                 }
                 usleep($this->_nWriteInterval);
 
                 $bError = $this->_updateQueue($aErrorMessage);
                 if ($bError) {
                     break;
                 }
             }
 
             if (!$bError) {
                 $read = array($this->_hSocket);
                 $null = NULL;
                 $nChangedStreams = @stream_select($read, $null, $null, 0, $this->_nSocketSelectTimeout);
                 if ($nChangedStreams === false) {
                     $this->_log('ERROR: Unable to wait for a stream availability.');
                     break;
                 } else if ($nChangedStreams > 0) {
                     $bError = $this->_updateQueue();
                     if (!$bError) {
                         $this->_aMessageQueue = array();
                     }
                 } else {
                     $this->_aMessageQueue = array();
                 }
             }
 
             $nRun++;
         }
     }
 
     /**
      * Returns messages in the message queue.
      *
      * When a message is successful sent or reached the maximum retry time is removed
      * from the message queue and inserted in the Errors container. Use the getErrors()
      * method to retrive messages with delivery error(s).
      *
      * @param  $bEmpty @type boolean @optional Empty message queue.
      * @return @type array Array of messages left on the queue.
      */
     public function getQueue($bEmpty = true)
     {
         $aRet = $this->_aMessageQueue;
         if ($bEmpty) {
             $this->_aMessageQueue = array();
         }
         return $aRet;
     }
 
     /**
      * Returns messages not delivered to the end user because one (or more) error
      * occurred.
      *
      * @param  $bEmpty @type boolean @optional Empty message container.
      * @return @type array Array of messages not delivered because one or more errors
      *         occurred.
      */
     public function getErrors($bEmpty = true)
     {
         $aRet = $this->_aErrors;
         if ($bEmpty) {
             $this->_aErrors = array();
         }
         return $aRet;
     }
 
     /**
      * Generate a binary notification from a device token and a JSON-encoded payload.
      *
      * @see http://tinyurl.com/ApplePushNotificationBinary
      *
      * @param  $sDeviceToken @type string The device token.
      * @param  $sPayload @type string The JSON-encoded payload.
      * @param  $nMessageID @type integer @optional Message unique ID.
      * @param  $nExpire @type integer @optional Seconds, starting from now, that
      *         identifies when the notification is no longer valid and can be discarded.
      *         Pass a negative value (-1 for example) to request that APNs not store
      *         the notification at all. Default is 86400 * 7, 7 days.
      * @return @type string A binary notification.
      */
     protected function _getBinaryNotification($sDeviceToken, $sPayload, $nMessageID = 0, $nExpire = 604800)
     {
         $nTokenLength = strlen($sDeviceToken);
         $nPayloadLength = strlen($sPayload);
 
         $sRet  = pack('CNNnH*', self::COMMAND_PUSH, $nMessageID, $nExpire > 0 ? time() + $nExpire : 0, self::DEVICE_BINARY_SIZE, $sDeviceToken);
         $sRet .= pack('n', $nPayloadLength);
         $sRet .= $sPayload;
 
         return $sRet;
     }
 
     /**
      * Parses the error message.
      *
      * @param  $sErrorMessage @type string The Error Message.
      * @return @type array Array with command, statusCode and identifier keys.
      */
     protected function _parseErrorMessage($sErrorMessage)
     {
         return unpack('Ccommand/CstatusCode/Nidentifier', $sErrorMessage);
     }
 
     /**
      * Reads an error message (if present) from the main stream.
      * If the error message is present and valid the error message is returned,
      * otherwhise null is returned.
      *
      * @return @type array|null Return the error message array.
      */
     protected function _readErrorMessage()
     {
         $sErrorResponse = @fread($this->_hSocket, self::ERROR_RESPONSE_SIZE);
         if ($sErrorResponse === false || strlen($sErrorResponse) != self::ERROR_RESPONSE_SIZE) {
             return;
         }
         $aErrorResponse = $this->_parseErrorMessage($sErrorResponse);
         if (!is_array($aErrorResponse) || empty($aErrorResponse)) {
             return;
         }
         if (!isset($aErrorResponse['command'], $aErrorResponse['statusCode'], $aErrorResponse['identifier'])) {
             return;
         }
         if ($aErrorResponse['command'] != self::ERROR_RESPONSE_COMMAND) {
             return;
         }
         $aErrorResponse['time'] = time();
         $aErrorResponse['statusMessage'] = 'None (unknown)';
         if (isset($this->_aErrorResponseMessages[$aErrorResponse['statusCode']])) {
             $aErrorResponse['statusMessage'] = $this->_aErrorResponseMessages[$aErrorResponse['statusCode']];
         }
         return $aErrorResponse;
     }
 
     /**
      * Checks for error message and deletes messages successfully sent from message queue.
      *
      * @param  $aErrorMessage @type array @optional The error message. It will anyway
      *         always be read from the main stream. The latest successful message
      *         sent is the lowest between this error message and the message that
      *         was read from the main stream.
      *         @see _readErrorMessage()
      * @return @type boolean True if an error was received.
      */
     protected function _updateQueue($aErrorMessage = null)
     {
         $aStreamErrorMessage = $this->_readErrorMessage();
         if (!isset($aErrorMessage) && !isset($aStreamErrorMessage)) {
             return false;
         } else if (isset($aErrorMessage, $aStreamErrorMessage)) {
             if ($aStreamErrorMessage['identifier'] <= $aErrorMessage['identifier']) {
                 $aErrorMessage = $aStreamErrorMessage;
                 unset($aStreamErrorMessage);
             }
         } else if (!isset($aErrorMessage) && isset($aStreamErrorMessage)) {
             $aErrorMessage = $aStreamErrorMessage;
             unset($aStreamErrorMessage);
         }
 
         $this->_log('ERROR: Unable to send message ID ' .
             $aErrorMessage['identifier'] . ': ' .
             $aErrorMessage['statusMessage'] . ' (' . $aErrorMessage['statusCode'] . ').');
 
         $this->disconnect();
 
         foreach($this->_aMessageQueue as $k => &$aMessage) {
             if ($k < $aErrorMessage['identifier']) {
                 unset($this->_aMessageQueue[$k]);
             } else if ($k == $aErrorMessage['identifier']) {
                 $aMessage['ERRORS'][] = $aErrorMessage;
             } else {
                 break;
             }
         }
 
         $this->connect();
 
         return true;
     }
 
     /**
      * Remove a message from the message queue.
      *
      * @param  $nMessageID @type integer The Message ID.
      * @param  $bError @type boolean @optional Insert the message in the Error container.
      * @throws ApnsPHP_Push_Exception if the Message ID is not valid or message
      *         does not exists.
      */
     protected function _removeMessageFromQueue($nMessageID, $bError = false)
     {
         if (!is_numeric($nMessageID) || $nMessageID <= 0) {
             throw new ApnsPHP_Push_Exception(
                 'Message ID format is not valid.'
             );
         }
         if (!isset($this->_aMessageQueue[$nMessageID])) {
             throw new ApnsPHP_Push_Exception(
                 "The Message ID {$nMessageID} does not exists."
             );
         }
         if ($bError) {
             $this->_aErrors[$nMessageID] = $this->_aMessageQueue[$nMessageID];
         }
         unset($this->_aMessageQueue[$nMessageID]);
     }
 }
 